<?php
/**
 * Repository.php 2020-3-24
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

declare(strict_types=1);

namespace Gambio\Shop\Attributes\Representation\SelectionHtml\Repository;

use Gambio\Shop\Attributes\ProductModifiers\Database\ValueObjects\AttributeModifierIdentifier;
use Gambio\Shop\Attributes\Representation\SelectionHtml\Exceptions\InvalidValueIdsSpecifiedException;
use Gambio\Shop\Attributes\Representation\SelectionHtml\Generators\ModifierHtmlGenerator;
use Gambio\Shop\Attributes\Representation\SelectionHtml\Repository\Readers\ReaderInterface;
use Gambio\Shop\Language\ValueObjects\LanguageId;
use Gambio\Shop\ProductModifiers\Modifiers\Collections\ModifierIdentifierCollectionInterface;
use Gambio\Shop\SellingUnit\Presentation\ValueObjects\OutOfStockMarkings;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SelectedQuantity;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitStockInterface;

/**
 * Class Repository
 * @package Gambio\Shop\Attributes\Representation\SelectionHtml\Repository
 */
class Repository implements RepositoryInterface
{
    /**
     * @var ReaderInterface
     */
    protected $reader;
    /**
     * @var OutOfStockMarkings
     */
    private $outOfStockMarkings;
    
    
    /**
     * Repository constructor.
     *
     * @param ReaderInterface    $reader
     * @param OutOfStockMarkings $outOfStockMarkings
     */
    public function __construct(ReaderInterface $reader, OutOfStockMarkings $outOfStockMarkings)
    {
        $this->reader             = $reader;
        $this->outOfStockMarkings = $outOfStockMarkings;
    }
    
    
    /**
     * @inheritDoc
     */
    public function selectionHtmlGenerator(
        ModifierIdentifierCollectionInterface $identifiers,
        LanguageId $languageId,
        SelectedQuantity $selectedQuantity,
        SellingUnitStockInterface $sellingUnitStock
    ): ModifierHtmlGenerator {
    
        $dtos = [];
        
        foreach ($identifiers as $identifier) {
            
            if ($identifier instanceof AttributeModifierIdentifier) {
                
                $dtos[] = $this->reader->selectionData($identifier, $languageId);
            }
        }
        
        return $this->createGenerator($dtos, $selectedQuantity, $sellingUnitStock);
    }
    
    
    /**
     * @param array                     $dtos
     *
     * @param SelectedQuantity          $selectedQuantity
     * @param SellingUnitStockInterface $sellingUnitStock
     *
     * @return ModifierHtmlGenerator
     */
    protected function createGenerator(
        array $dtos,
        SelectedQuantity $selectedQuantity,
        SellingUnitStockInterface $sellingUnitStock
    ): ModifierHtmlGenerator {
        
        return new ModifierHtmlGenerator($dtos, $selectedQuantity, $sellingUnitStock, $this->outOfStockMarkings);
    }
}